// EditorView.cpp : implementation of the CEditorView class
//

#include "stdafx.h"
#include <string>
#include <cassert>
#include <algorithm>
#include "Quincy.h"
#include "MainFrm.h"
#include "ChildFrm.h"
#include "EditorDoc.h"
#include "EditorView.h"
#include "QuincyPrintDialog.h"
#include "Beautifier.h"
#include "QuincyDoc.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

#define PRINTMARGIN 2


/////////////////////////////////////////////////////////////////////////////
// EditorSelection

EditorSelection::EditorSelection(CEditorView& view)
					: sel::Selection<ViewIndex, CEditorView>(view)
{
}

/////////////////////////////////////////////////////////////////////////////
// ViewIndex

ViewIndex& ViewIndex::operator=(const ViewIndex& view)
{
	if (this != &view)	{
		pView = view.pView;
		row = view.row;
		column = view.column;
	}
	return *this;
}
bool ViewIndex::operator==(const ViewIndex& view) const
{
	return pView == view.pView && row == view.row && column == view.column;
}
bool ViewIndex::operator<(const ViewIndex& view) const
{
	if (pView == 0 && view.pView != 0)
		return true;
	if (pView != 0 && view.pView == 0)
		return false;
	if (row == view.row)
		return column < view.column;
	return row < view.row;
}


/////////////////////////////////////////////////////////////////////////////
// CEditorView

CFindReplaceDialog* CEditorView::m_pFindDlg;
std::string CEditorView::m_findtext;
std::string CEditorView::m_replacetext;
int CEditorView::m_nfrflags;
CEditorView* CEditorView::pEditor;

bool CEditorView::CodeElement::operator<(const struct CodeElement& ce) const
{
	if (line == ce.line)
		return (charpos < ce.charpos);
	return line < ce.line;
}


IMPLEMENT_DYNCREATE(CEditorView, CView)

static UINT WM_FINDREPLACE = ::RegisterWindowMessage(FINDMSGSTRING);

BEGIN_MESSAGE_MAP(CEditorView, CView)
	//{{AFX_MSG_MAP(CEditorView)
	ON_WM_VSCROLL()
	ON_WM_HSCROLL()
	ON_WM_KILLFOCUS()
	ON_WM_SETFOCUS()
	ON_WM_LBUTTONDOWN()
	ON_WM_LBUTTONDBLCLK()
	ON_WM_KEYDOWN()
	ON_WM_KEYUP()
	ON_WM_CHAR()
	ON_WM_LBUTTONUP()
	ON_WM_MOUSEMOVE()
	ON_WM_TIMER()
	ON_UPDATE_COMMAND_UI(ID_EDIT_CUT, OnUpdateEditCut)
//	ON_UPDATE_COMMAND_UI(ID_EDIT_COPY, OnUpdateEditCopy)
	ON_UPDATE_COMMAND_UI(ID_EDIT_PASTE, OnUpdateEditPaste)
	ON_COMMAND(ID_EDIT_CUT, OnEditCut)
	ON_COMMAND(ID_EDIT_COPY, OnEditCopy)
	ON_COMMAND(ID_EDIT_PASTE, OnEditPaste)
	ON_UPDATE_COMMAND_UI(ID_EDIT_CLEAR, OnUpdateEditClear)
	ON_COMMAND(ID_EDIT_CLEAR, OnEditClear)
	ON_COMMAND(ID_EDIT_SELECT_ALL, OnEditSelectAll)
	ON_COMMAND(ID_EDIT_FIND, OnEditFind)
	ON_COMMAND(ID_EDIT_REPLACE, OnEditReplace)
	ON_COMMAND(ID_EDIT_FIND_NEXT, OnEditFindNext)
	ON_UPDATE_COMMAND_UI(ID_EDIT_FIND_NEXT, OnUpdateEditFindNext)
	ON_COMMAND(ID_EDIT_UNDO, OnEditUndo)
	ON_COMMAND(ID_EDIT_REDO, OnEditRedo)
	ON_COMMAND(ID_RECORD, OnRecord)
	ON_COMMAND(ID_STOPRECORD, OnStopRecord)
	ON_WM_SETCURSOR()
	ON_UPDATE_COMMAND_UI(ID_PLAYBACK, OnUpdatePlayback)
	ON_UPDATE_COMMAND_UI(ID_RECORD, OnUpdateRecord)
	ON_UPDATE_COMMAND_UI(ID_STOPRECORD, OnUpdateStopRecord)
	ON_COMMAND(ID_PLAYBACK, OnPlayback)
	ON_COMMAND(ID_FILE_OPEN, OnFileOpen)
	ON_COMMAND(ID_FILE_PRINT, OnFilePrint)
	ON_UPDATE_COMMAND_UI(ID_FILE_SAVE_ALL, OnUpdateFileSaveAll)
	ON_UPDATE_COMMAND_UI(ID_BRACEMATCH, OnUpdateBracematch)
	ON_COMMAND(ID_BRACEMATCH, OnBracematch)
	ON_UPDATE_COMMAND_UI(ID_ASTYLE, OnUpdateBeautify)
	ON_COMMAND(ID_ASTYLE, OnBeautify)
	//}}AFX_MSG_MAP
	// Standard printing commands
//	ON_COMMAND(ID_FILE_PRINT, CView::OnFilePrint)
	ON_COMMAND(ID_FILE_PRINT_DIRECT, CView::OnFilePrint)
	ON_COMMAND(ID_FILE_PRINT_PREVIEW, CView::OnFilePrintPreview)
	ON_REGISTERED_MESSAGE(WM_FINDREPLACE, OnFindReplace)
END_MESSAGE_MAP()

// C/C++ keywords
std::string CEditorView::ckeys[] = {
		"#define",
		"#elif",
		"#else",
		"#endif",
		"#error",
		"#if",
		"#ifdef",
		"#ifndef",
		"#include",
		"#line",
		"#pragma",
		"#undef",
		"asm",
		"auto",
		"bool",
		"break",
		"case",
		"char",
		"const",
		"continue",
		"default",
		"do",
		"double",
		"else",
		"enum",
		"extern",
		"float",
		"for",
		"goto",
		"if",
		"int",
		"long",
		"register",
		"return",
		"short",
		"signed",
		"sizeof",
		"static",
		"struct",
		"switch",
		"typedef",
		"union",
		"unsigned",
		"void",
		"volatile",
		"while"
};
// C++ keywords
std::string CEditorView::cppkeys[] = {
		"and",
		"and_eq",
		"bitand",
		"bitor",
		"catch",
		"class",
		"compl",
		"const_cast",
		"delete",
		"dynamic_cast",
		"explicit",
		"false",
		"friend",
		"inline",
		"mutable",
		"namespace",
		"new",
		"not",
		"not_eq",
		"operator",
		"or",
		"or_eq",
		"private",
		"protected",
		"public",
		"reinterpret_cast",
		"static_cast",
		"template",
		"this",
		"throw",
		"true",
		"try",
		"typeid",
		"typename",
		"using",
		"virtual",
		"wchar_t",
		"xor",
		"xor_e"
};

/////////////////////////////////////////////////////////////////////////////
// CEditorView construction/destruction

CEditorView::CEditorView()
{
	m_pscreenfont = 0;
	m_pConsoleApp = 0;
	m_pClearBrush = 0;
	m_pMarginBrush = 0;
	m_pFindDlg = 0;
	m_fontheight = theApp.FontHeight();
	m_fontwidth = theApp.FontWidth();
	m_fontweight = theApp.FontWeight();
	m_leftcolumn = 0;
	m_toprow = 0;
	m_bottomrow = 0;
	m_windowrows = 0;
	m_windowcolumns = 0;
	m_currentrow = 0;
	m_currentcolumn = 0;
	m_margin = theApp.GetMargin() * 8;// m_fontwidth;
	m_ctrldown = false;
	m_shiftdown = false;
	m_pEditorSelection = new EditorSelection(*this);
	m_mousemarking = false;
	m_dragging = false;
	m_recording = false;
	m_selectionmarked = false;
	((CMainFrame*)theApp.m_pMainWnd)->SetRowColumn(1, 1);
	m_hOldCursor = 0;
	m_hMoveCursor = theApp.LoadCursor(IDC_POINTER_MOVE);
	m_hCopyCursor = theApp.LoadCursor(IDC_POINTER_COPY);
	m_hRecordCursor = theApp.LoadCursor(IDC_POINTER_RECORD);
	m_hArrowCursor	= theApp.LoadStandardCursor(IDC_ARROW);
	m_pFindDlg = 0;
	m_nfrflags = FR_DOWN;
	m_buildcomments = true;
	m_linelength = 0;
	m_linecount = 0;
	m_linescount = 0;
	m_charcount = 0;
	m_inccomment = false;
	m_incppcomment = false;
	m_instringliteral = false;
	m_incharliteral = false;
	m_found = false;
	m_wasfound = false;
	m_bchanged = false;
}

CEditorView::~CEditorView()
{
	delete m_pConsoleApp;
	delete m_pClearBrush;
	delete m_pMarginBrush;
	delete m_pEditorSelection;
	delete m_pFindDlg;
	delete m_pscreenfont;
	DeleteMacros();
}

BOOL CEditorView::PreCreateWindow(CREATESTRUCT& cs)
{
	cs.style |= WS_HSCROLL | WS_VSCROLL;
	return CView::PreCreateWindow(cs);
}

void CEditorView::RebuildFont()
{
	delete m_pscreenfont;
	m_pscreenfont = new CFont;
	m_fontheight = theApp.FontHeight();
	m_fontwidth = theApp.FontWidth();
	m_fontweight = theApp.FontWeight();
	theApp.EditorScreenFont(m_pscreenfont, m_fontheight,m_fontwidth,m_fontweight);
	CClientDC cdc(this);
	CFont* pOldFont = cdc.SelectObject(m_pscreenfont);
	TEXTMETRIC tm;
	cdc.GetTextMetrics(&tm);
	m_fontheight = tm.tmHeight;
	m_fontwidth = tm.tmMaxCharWidth;
	theApp.SetFontHeight(m_fontheight);
	theApp.SetFontWidth(m_fontwidth);
	theApp.SetFontWeight(m_fontweight);
	cdc.SelectObject(pOldFont);
	CreateSolidCaret(2, m_fontheight);
	SetScrollPosition(SB_VERT, m_toprow);
	SetCaretPosition();
	AdjustScroll();
}

BOOL CEditorView::Create(LPCTSTR lpszClassName, LPCTSTR lpszWindowName, DWORD dwStyle, const RECT& rect, CWnd* pParentWnd, UINT nID, CCreateContext* pContext) 
{
	bool rtn = CWnd::Create(lpszClassName, lpszWindowName, dwStyle, rect, pParentWnd, nID, pContext);

	// --- register with parent frame class 
	CChildFrame* pParent = (CChildFrame*)GetParentFrame();
	ASSERT(pParent != 0);
	pParent->RegisterView(this);

	m_pClearBrush = new CBrush(RGB(255,255,255));
	m_pMarginBrush = new CBrush(RGB(192,192,192));

	RebuildFont();

	return rtn;
}

/////////////////////////////////////////////////////////////////////////////
// CEditorView drawing

void CEditorView::AddColorToTable(int line, int charpos, COLORREF clr)
{
	CodeElement ce = {line, charpos, clr};

	std::set<CodeElement>::iterator it = m_contexttable.find(ce);
	if (it != m_contexttable.end())
		m_contexttable.erase(it);
	m_contexttable.insert(ce);
	m_prevcolor = clr;
}

int CEditorView::GetNextSourceChar()
{
	if (m_charcount >= m_linelength)	{
		// --- end of line
		KeywordSearch();
		if (m_incppcomment)	{
			// --- end of C++ comment is at end of line
			m_incppcomment = false;
			m_prevcolor = theApp.NormalColor();
		}
		m_incharliteral = false;
		m_charcount = 0;
		if (m_linecount >= m_linescount)
			return -1;
		AddColorToTable(m_linecount, 0, m_prevcolor);
		m_linestr = GetDocument()->textline(m_linecount++);
		m_linelength = m_linestr.length();
	}
	return m_linestr[m_charcount++];
}

void CEditorView::KeywordSearch()
{
	if (kword.length() > 0)	{
		if (keywords.find(kword) != keywords.end())	{
			int ln = m_linecount-1;
			int col = m_charcount - 1;
			if (m_charcount == m_linelength && m_charcount == kword.length())
				col++;		// keyword is only thing on line
			AddColorToTable(ln, col-kword.length(), theApp.KeywordColor());
			AddColorToTable(ln, col, theApp.NormalColor());
		}
		kword.erase();
	}
}

void CEditorView::BuildContextHighlightTable()
{
	m_linecount = 0;
	m_linescount = GetDocument()->linecount();
	m_charcount = 0;

	m_linestr.erase();

	m_linelength = 0;
	m_contexttable.clear();

	m_inccomment = false;
	m_incppcomment = false;
	m_instringliteral = false;
	m_incharliteral = false;

	m_prevcolor = theApp.NormalColor();

	char ch;
	while ((ch = GetNextSourceChar()) != -1)	{
		if (m_incppcomment)
			// --- in a C++ comment, scan until end of line
			continue;
		if (!m_inccomment && !m_instringliteral && !m_incharliteral)	{
			// --- not currently in C comment, string literal, or char literal
			if (ch == '"')	{
				// --- start of a string literal
				KeywordSearch();
				m_instringliteral = true;
				AddColorToTable(m_linecount-1, m_charcount-1, theApp.StringColor());
			}
			else if (ch == '\'')	{
				// --- start of a character literal
				m_incharliteral = true;
			}
			else if (ch == '/')	{
				// --- possible start of a comment
				KeywordSearch();
				if ((ch = GetNextSourceChar()) == -1)
					break;
				if (m_charcount < 2)
					// --- ch is the first character of a line
					continue;
				if (ch == '/')
					// --- start of a C++ comment
					m_incppcomment = true;
				if (ch == '*')
					// --- begin token ( /* ) of a C comment
					m_inccomment = true;
				if (m_incppcomment || m_inccomment)
					// --- save the starting position of the comment
					AddColorToTable(m_linecount-1, m_charcount-2, theApp.CommentColor());
			}
			else	{
				// --- potential keyword
				if ((kword.length() == 0 && ch == '#') || isalpha(ch) || ch == '_')
					kword += ch;	// collect a word
				else
					KeywordSearch();
			}
			continue;
		}
		if (m_instringliteral)	{
			// --- parsing a string literal
			if (ch == '\\')	{
				// --- bypass escape sequences
				if ((ch = GetNextSourceChar()) == -1)
					break;
				continue;
			}
			if (ch == '"')	{
				// --- end of string literal
				m_instringliteral = false;
				AddColorToTable(m_linecount-1, m_charcount, theApp.NormalColor());
			}
		}
		else if (m_incharliteral)	{
			// --- parsing a character literal
			if (ch == '\\')	{
				// --- bypass escape sequences
				if ((ch = GetNextSourceChar()) == -1)
					break;
				continue;
			}
			if (ch == '\'')
				// --- end of character literal
				m_incharliteral = false;
		}
		else if (m_inccomment)	{
			while (ch == '*')	{
				if ((ch = GetNextSourceChar()) == -1)
					break;
				if (m_charcount > 1 && ch == '/')	{
					// --- end token ( * / ) of a C comment
					m_inccomment = false;
					AddColorToTable(m_linecount-1, m_charcount-1, theApp.NormalColor());
					break;
				}
			}
			if (ch == -1)
				break;
		}
	}
#ifdef _DEBUG
//	DumpContextTable();
#endif
}

void CEditorView::DisplayText(CDC* pDC, int beg, int len, int line, COLORREF color)
{
	std::string str = GetDocument()->textline(line).substr(beg, len);
	CRect rc;
	GetClientRect(&rc);
	rc.left = m_margin;

	int length = str.length();
	// ---- hide the tab control characters in the text line
	for (int i = 0; i < length; i++)	{
		if (str[i] == '\t')
			str[i] = ' ';
	}

	pDC->SetBkColor(theApp.BackgroundColor());
	pDC->SetTextColor(color);

	int x = m_margin + beg * m_fontwidth - m_leftcolumn;
	int top = line * m_fontheight - m_toprow;
	pDC->ExtTextOut(x, top, ETO_CLIPPED, &rc, str.c_str(), len, 0);
}


void CEditorView::DisplaySourceText(CDC* pDC, int line)
{
	const std::string& str = GetDocument()->textline(line);

	if (theApp.SyntaxColorsOption() && IsSourceCodeFile())	{
		CodeElement ce;
		ce.line = line;
		ce.charpos = 0;
		std::set<CodeElement>::const_iterator it = m_contexttable.find(ce);
		ASSERT(it != m_contexttable.end()); // must be an entry for char 0 of every line
		// Always asserts here on exit from debugger
		do	{
			COLORREF clr = (*it).color;
			int beg = ce.charpos;
			int end = str.length();
			if (++it != m_contexttable.end())	{
				ce = *it;
				if (ce.line == line)
					end = ce.charpos;
			}
			DisplayText(pDC, beg, end-beg, line, clr);
		} while (ce.line == line && it != m_contexttable.end());
	}
	else
		// --- not a source code file or syntax color option not selected
		DisplayText(pDC, 0, str.length(), line, theApp.NormalColor());
}

void CEditorView::InsertLineIntoScreen(CDC* pDC, unsigned int line)
{
	HideCaret();

	if (m_buildcomments)	{
		if (theApp.SyntaxColorsOption() && IsSourceCodeFile())
			BuildContextHighlightTable();
		m_buildcomments = false;
	}

	const std::string& str = GetDocument()->textline(line);
	int len = str.length();

	// --- display the text on the line
    CFont* pOldFont = pDC->SelectObject(m_pscreenfont);

	DisplaySourceText(pDC, line);

	int top = line * m_fontheight - m_toprow;

	// --- pad the line with spaces
	len -= m_leftcolumn / m_fontwidth;
	CRect rc(m_margin, top, m_windowcolumns, top + m_fontheight);
	rc.left += len * m_fontwidth;
	pDC->FillRect(&rc, m_pClearBrush);
    pDC->SelectObject(pOldFont);

	if (m_selectionmarked && line >= m_start.row && line <= m_stop.row)	{
		// ---- this line is in the selection, need to do some inverting in the bitmap
		int left = m_margin - m_leftcolumn;
		if (line == m_start.row)
			left += m_start.column * m_fontwidth;
		int right = m_margin + (line < m_stop.row ? len : m_stop.column) * m_fontwidth;
		// for blank line display as if there is one space character
		if (left == right && right == m_margin)
			right += m_fontwidth;
		int wd = right - left;
		if (wd)
			pDC->PatBlt(left, top, wd, m_fontheight, DSTINVERT);
	}
	ShowCaret();
	UpdateTitle(GetDocument());
}

bool CEditorView::IsSourceFile(const CString& ext)
{
	CDocument* pDoc = GetDocument();
	if (pDoc != 0)	{
		CString strPath = pDoc->GetPathName();
		return strPath.Right(ext.GetLength()).CompareNoCase(ext) == 0;
	}
	return false;
}

void CEditorView::OnDraw(CDC* pDC) 
{
	if (GetDocument() == 0)
		return;
	delete m_pClearBrush;
	m_pClearBrush = new CBrush(theApp.BackgroundColor());
	if (keywords.empty())	{
		int i;
		bool isc = IsCFile();
		bool ish = IsHFile();
		bool iscpp = IsCPPFile();
		bool isrc = IsRCFile();
		if (isrc)
			for (i = 0; i < 12; i++)
				keywords.insert(ckeys[i]);
		if (isc || ish || iscpp)
			for (i = 0; i < sizeof(ckeys) / sizeof(std::string); i++)
				keywords.insert(ckeys[i]);
		if (iscpp || ish)
			for (i = 0; i < sizeof(cppkeys) / sizeof(std::string); i++)
				keywords.insert(cppkeys[i]);
	}
	HideCaret();
	CRect rc;
	GetClientRect(&rc);
	m_windowrows = rc.bottom;
	m_windowcolumns = rc.right;

	unsigned int linecount = GetDocument()->linecount();

	m_bottomrow = linecount * m_fontheight;
	m_linewidth = GetDocument()->linewidth() * m_fontwidth;

	SCROLLINFO sivert = {
		sizeof (SCROLLINFO),
		SIF_RANGE,
		0, m_bottomrow,
	};
	SCROLLINFO sihorz = {
		sizeof (SCROLLINFO),
		SIF_RANGE,
		0, m_linewidth,
	};

	SetScrollInfo(SB_VERT, &sivert);
	SetScrollInfo(SB_HORZ, &sihorz);

	unsigned int line = m_toprow / m_fontheight;
	unsigned int windowlines = (m_toprow + m_windowrows + m_fontheight) / m_fontheight;
	linecount = min(linecount, windowlines);

	if ((m_selectionmarked = IsSelectionMarked()) == true)
		m_pEditorSelection->GetSelectionMarkers(m_start, m_stop);
	while (line < linecount)
		InsertLineIntoScreen(pDC, line++);

	unsigned int lastrow = line * m_fontheight - m_toprow;	// last window row drawn
	if (lastrow < m_windowrows)	{
		// --- pad to the bottom of the window with spaces
		CRect crc(rc);
		crc.left = m_margin;
		crc.top = lastrow;
		pDC->FillRect(&crc, m_pClearBrush);
	}
	delete m_pClearBrush;
	m_pClearBrush = 0;
	PadMargin();
}

void CEditorView::PadMargin()
{
	// ----- pad the left margin with a gray tone
	CDC* pDC = GetDC();	// to pad the whole margin to avoid having to erase margin characters
	CRect rc;
	GetClientRect(&rc);
	rc.right = m_margin;
	pDC->FillRect(&rc, m_pMarginBrush);
	ShowCaret();
}

/////////////////////////////////////////////////////////////////////////////
// CEditorView printing

BOOL CEditorView::OnPreparePrinting(CPrintInfo* pInfo)
{
	delete pInfo->m_pPD;					// framework has one built on heap
	pInfo->m_pPD = new CQuincyPrintDialog;	// framework will delete this object
	static_cast<CQuincyPrintDialog*>(pInfo->m_pPD)->printlinenumbers = theApp.PrintLineNumbers();
	return DoPreparePrinting(pInfo);
}

void CEditorView::OnBeginPrinting(CDC* pDC, CPrintInfo* pInfo)
{
	printlinenos = static_cast<CQuincyPrintDialog*>(pInfo->m_pPD)->printlinenumbers;

    int nHeight = -((pDC->GetDeviceCaps(LOGPIXELSY) * 10) / 72);

    m_printerfont.CreateFont(nHeight, 0, 0, 0, FW_NORMAL, 0, 0, 0,
        DEFAULT_CHARSET, OUT_CHARACTER_PRECIS, CLIP_CHARACTER_PRECIS,
        DEFAULT_QUALITY, DEFAULT_PITCH | FF_DONTCARE, "Courier");

    TEXTMETRIC tm;
    CFont* pOldFont = pDC->SelectObject(&m_printerfont);
    pDC->GetTextMetrics(&tm);
    m_cyPrinter = tm.tmHeight + tm.tmExternalLeading;
    pDC->SelectObject(pOldFont);

    m_nLinesPerPage =(pDC->GetDeviceCaps(VERTRES) -
       (m_cyPrinter * (3 + (2 * PRINTMARGIN)))) / m_cyPrinter;
    UINT nMaxPage = max(1, (GetDocument()->linecount() + (m_nLinesPerPage - 1)) /
        m_nLinesPerPage);
    pInfo->SetMaxPage(nMaxPage);
}

void CEditorView::OnPrint (CDC* pDC, CPrintInfo* pInfo)
{
    PrintPageHeader(pDC, pInfo->m_nCurPage);
    PrintPage(pDC, pInfo->m_nCurPage);
}

void CEditorView::OnEndPrinting(CDC* /*pDC*/, CPrintInfo* pInfo)
{
    m_printerfont.DeleteObject ();
	bool plno = static_cast<CQuincyPrintDialog*>(pInfo->m_pPD)->printlinenumbers;
	theApp.SetPrintLineNumbers(plno);
}

void CEditorView::PrintPageHeader(CDC* pDC, UINT nPageNumber)
{
    CString strHeader = GetDocument()->GetTitle();
	CString strPageNumber;
	strPageNumber.Format(" (Page %d)", nPageNumber);
	strHeader += strPageNumber;

    UINT y = m_cyPrinter * PRINTMARGIN;
    CFont* pOldFont = pDC->SelectObject(&m_printerfont);
    pDC->TextOut(0, y, strHeader);
    pDC->SelectObject(pOldFont);
}

void CEditorView::PrintPage (CDC* pDC, UINT nPageNumber)
{
	int lines = GetDocument()->linecount();
    if (GetDocument()->linecount() != 0) {
        UINT nStart = (nPageNumber - 1) * m_nLinesPerPage;
        UINT nEnd = min(lines - 1, nStart + m_nLinesPerPage - 1);

        CFont* pOldFont = pDC->SelectObject(&m_printerfont);

        for (int i = nStart; i <= nEnd; i++) {
			std::string str = GetDocument()->textline(i);
			for (int j = 0; j < str.length(); j++)
				if (str[j] == '\t')
					str[j] = ' ';
            int y = ((i - nStart) + PRINTMARGIN + 3) * m_cyPrinter;
			CString line;
			if (printlinenos)
				line.Format("%-4d ", i + 1);
			line += str.c_str();
            pDC->TextOut(0, y, line);
        }
        pDC->SelectObject(pOldFont);
    }
}

/////////////////////////////////////////////////////////////////////////////
// CEditorView diagnostics

#ifdef _DEBUG
void CEditorView::AssertValid() const
{
	CView::AssertValid();
}

void CEditorView::Dump(CDumpContext& dc) const
{
	CView::Dump(dc);
}

CEditorDoc* CEditorView::GetDocument() // non-debug version is inline
{
	ASSERT(m_pDocument == 0 || m_pDocument->IsKindOf(RUNTIME_CLASS(CEditorDoc)));
	return (CEditorDoc*)m_pDocument;
}
#endif //_DEBUG

////////////////// scrolling and paging functions

void CEditorView::SetScrollPosition(int nBar, UINT nPos)
{
	SCROLLINFO si = {
		sizeof(SCROLLINFO),
		SIF_POS,
		0,0,0,
		nPos
	};
	SetScrollInfo(nBar, &si);
}

void CEditorView::VScrollTo(UINT nPos)
{
	if (m_toprow != nPos)	{
		m_toprow = nPos;
		SetScrollPosition(SB_VERT, nPos);
		SetCaretPosition();
		Invalidate(false);
	}
}
void CEditorView::HScrollTo(UINT nPos)
{
	if (m_leftcolumn != nPos)	{
		m_leftcolumn = nPos;
		SetScrollPosition(SB_HORZ, nPos);
		SetCaretPosition();
		Invalidate(false);
	}
}
void CEditorView::ScrollDown()
{
	if (m_toprow > 0)	{
		int nsize = m_toprow < vscrollsize() ? m_toprow : vscrollsize();
		CRect rc;
		GetClientRect(&rc);
		rc.left += m_margin;
		ScrollWindow(0, nsize, &rc, &rc);
		m_toprow -= nsize;
		UpdateWindow();
		SetScrollPosition(SB_VERT, m_toprow);
		SetCaretPosition();
	}
}
void CEditorView::ScrollUp()
{
	if (GetScrollPos(SB_VERT) < m_toprow + vscrollsize())	{
		CRect rc;
		GetClientRect(&rc);
		rc.left += m_margin;
		ScrollWindow(0, -vscrollsize(), &rc, &rc);
		m_toprow += vscrollsize();
		UpdateWindow();
		SetScrollPosition(SB_VERT, m_toprow);
		SetCaretPosition();
	}
}
void CEditorView::ScrollLeft()
{
	if (m_leftcolumn > 0)	{
		int nsize = m_leftcolumn < hscrollsize() ? m_leftcolumn : hscrollsize();
		CRect rc;
		GetClientRect(&rc);
		rc.left += m_margin;
		ScrollWindow(nsize, 0, 0, &rc);
		m_leftcolumn -= nsize;
		UpdateWindow();
		SetScrollPosition(SB_HORZ, m_leftcolumn);
		SetCaretPosition();
	}
}
void CEditorView::ScrollRight()
{
	if (GetScrollPos(SB_HORZ) < m_leftcolumn + hscrollsize())	{
		CRect rc;
		GetClientRect(&rc);
		rc.left += m_margin;
		ScrollWindow(-hscrollsize(), 0, 0, &rc);
		m_leftcolumn += hscrollsize();
		UpdateWindow();
		SetScrollPosition(SB_HORZ, m_leftcolumn);
		SetCaretPosition();
	}
}
void CEditorView::PageLeft()
{
	if (m_leftcolumn > 0)	{
		m_leftcolumn -= hpagesize();
		if (m_leftcolumn < 0)
			m_leftcolumn = 0;
		CRect rc;
		GetClientRect(&rc);
		rc.left += m_margin;
		SetScrollPosition(SB_HORZ, m_leftcolumn);
		ScrollWindow(hpagesize(), 0, &rc, &rc);
		SetCaretPosition(false, true);
	}
}
void CEditorView::PageRight()
{
	if (GetScrollPos(SB_HORZ) < m_leftcolumn + hpagesize())	{
		m_leftcolumn += hpagesize();
		CRect rc;
		GetClientRect(&rc);
		rc.left += m_margin;
		ScrollWindow(-hpagesize(), 0, &rc, &rc);
		SetScrollPosition(SB_HORZ, m_leftcolumn);
		SetCaretPosition(false, true);
	}
}
void CEditorView::PageHome()
{
	VScrollTo(0);
	HScrollTo(0);
	m_currentrow = 0;
	m_currentcolumn = 0;
	SetCaretPosition(false, true);
}
void CEditorView::PageEnd()
{
	int nto = m_bottomrow - vpagesize();
	VScrollTo(nto > 0 ? nto : 0);
	HScrollTo(0);
	m_currentrow = GetDocument()->linecount();
	m_currentcolumn = 0;
	SetCaretPosition(false, true);
}
void CEditorView::PageUp()
{
	if (m_toprow > 0)	{
		m_toprow -= vpagesize();
		if (m_toprow < 0)
			m_toprow = 0;
		Invalidate(false);
		SetScrollPosition(SB_VERT, m_toprow);
		SetCaretPosition(false, true);
	}
}
void CEditorView::PageDown()
{
	if (m_toprow + vpagesize() < m_bottomrow)	{
		m_toprow += vpagesize();
		Invalidate(false);
		SetScrollPosition(SB_VERT, m_toprow);
		SetCaretPosition(false, true);
	}
}

////////////////////////////////////////////////
// Functions for getting current position into view

void CEditorView::AdjustScroll()
{
	int col = m_currentcolumn * m_fontwidth;
	while (col > m_windowcolumns + m_leftcolumn - m_fontwidth)
		PageRight();
	while (col < m_leftcolumn)
		PageLeft();
	int row = m_currentrow * m_fontheight;
	while (row > m_windowrows + m_toprow - m_fontheight)
		ScrollUp();
	while (row < m_toprow)
		ScrollDown();
}

void CEditorView::AdjustScrollQuietly()
{
	int col = m_currentcolumn * m_fontwidth;
	while (col > m_windowcolumns + m_leftcolumn - m_fontwidth)	{
		if (GetScrollPos(SB_HORZ) < m_leftcolumn + hpagesize())
			m_leftcolumn += hpagesize();
	}
	while (col < m_leftcolumn)	{
		if (m_leftcolumn > 0)	{
			m_leftcolumn -= hpagesize();
			if (m_leftcolumn < 0)
				m_leftcolumn = 0;
		}
	}
	CurrentLineIntoView();
}

void CEditorView::CurrentLineIntoView()
{
	int row = m_currentrow * m_fontheight;
	if (row > m_windowrows + m_toprow - m_fontheight)
		while (row > m_windowrows/2 + m_toprow - m_fontheight)
			m_toprow += vscrollsize();
	if (row < m_toprow)	{
		while (row < m_toprow + m_windowrows/2)	{
			int nsize = m_toprow < vscrollsize() ? m_toprow : vscrollsize();
			m_toprow -= nsize;
			if (m_toprow <= 0)	{
				m_toprow = 0;
				break;
			}
		}
	}
	SetCaretPosition();
	SetScrollPosition(SB_VERT, m_toprow);
	Invalidate(false);
}

////////////////////////////////////////////////
// caret movement functions

void CEditorView::CaretDown()
{
	if (m_currentrow + 1 < GetDocument()->linecount())	{
		m_currentrow++;
		AdjustScroll();
		SetCaretPosition(false, true);
	}
}
void CEditorView::CaretUp()
{
	if (m_currentrow > 0)	{
		--m_currentrow;
		AdjustScroll();
		SetCaretPosition(false, true);
	}
}

void CEditorView::CaretRight()
{
	const std::string& str = GetDocument()->textline(m_currentrow);
	if (m_currentcolumn == str.length() && m_currentrow + 1 < GetDocument()->linecount())	{
		m_currentcolumn = 0;
		m_currentrow++;
		if (m_leftcolumn != 0)
			HScrollTo(0);
	}
	else
		m_currentcolumn++;
	AdjustScroll();
	SetCaretPosition(true, true);
}
void CEditorView::CaretLeft()
{
	if (m_currentcolumn == 0)	{
		if (m_currentrow == 0)
			return;
		const std::string& str = GetDocument()->textline(--m_currentrow);
		m_currentcolumn = str.length();
	}
	else
		--m_currentcolumn;
	AdjustScroll();
	SetCaretPosition(false, true);
}

static bool iswordchar(char ch)
{
	return isalpha(ch) || isdigit(ch) || ch == '_';
}

void CEditorView::WordRight()
{
	int linect = GetDocument()->linecount();
	if (m_currentrow == linect)
		return;
	const std::string* str = &GetDocument()->textline(m_currentrow);
	if (str->empty())	{
		// --- sitting on a blank line
		CaretRight();
		return;
	}
	// --- move out of the current word
	bool testwdch = iswordchar((*str)[m_currentcolumn]);
	while (!isspace((*str)[m_currentcolumn]) && 
			iswordchar((*str)[m_currentcolumn]) == testwdch)	{
		if (m_currentrow == linect-1 && m_currentcolumn == str->length())
			return;		// at end of file
		CaretRight();
		if (m_currentcolumn == 0)	{
			if (m_currentrow == linect)
				return;
			str = &GetDocument()->textline(m_currentrow);
			break;
		}
	}
	// --- move to the next word
	while (!str->empty() && isspace((*str)[m_currentcolumn]))	{
		CaretRight();
		if (m_currentcolumn == 0)	{
			if (m_currentrow == linect)
				return;
			str = &GetDocument()->textline(m_currentrow);
		}
	}
}
void CEditorView::WordLeft()
{
	CaretLeft();
	if (m_currentrow == 0 && m_currentcolumn == 0)
		return;
	const std::string* str = &GetDocument()->textline(m_currentrow);
	if (str->empty())
		return;
	// --- move back out of spaces to previous word
	while (isspace((*str)[m_currentcolumn]))	{
		if (m_currentcolumn == 0)
			return;
		CaretLeft();
	}

	bool testwdch = iswordchar((*str)[m_currentcolumn]);
	while (iswordchar((*str)[m_currentcolumn]) == testwdch && 
				!isspace((*str)[m_currentcolumn]))	{
		if (m_currentcolumn == 0)
				return;
		CaretLeft();
	}
	CaretRight();
}
void CEditorView::LineEnd()
{
	const std::string& str = GetDocument()->textline(m_currentrow);
	m_currentcolumn = str.length();
	AdjustScroll();
	SetCaretPosition(false, true);
}
void CEditorView::LineHome()
{

	int lno = CurrentLineNumber();
	const std::string& line = GetDocument()->textline(lno-1);
	int len = line.length();
	int i;
	for (i = 0; i < len; i++)
		if (line[i] != '\t' && line[i] != ' ')
			break;
	m_currentcolumn = (i < len && m_currentcolumn > i) ? i : 0;
	if (m_leftcolumn > m_currentcolumn * m_fontwidth)
		HScrollTo(0);
	SetCaretPosition(false, true);
}

/////////////////////////////////////////////////////////////////////////
// Editing functions

void CEditorView::DeleteWord()
{
	m_buildcomments = true;
	if (IsSelectionMarked())
		DeleteSelection();
	CEditorDoc* pDoc = GetDocument();
	ASSERT(pDoc != 0);
	const std::string line = pDoc->textline(m_currentrow);

	int len = line.length();

	bool blankline = len == 0 && pDoc->linecount() < m_windowrows / m_fontheight;

	bool endofline = m_currentcolumn == len;

	if (blankline || endofline)	{
		DeleteCharacter();
		return;
	}

	int i = m_currentcolumn;
	if (isspace(line[i]))	{
		while (i < len && isspace(line[i]))	{
			pDoc->deletechar(m_currentrow, m_currentcolumn, i == m_currentcolumn);
			i++;
		}
	}
	else if (!isalpha(line[i]) && line[i] != '_')
		pDoc->deletechar(m_currentrow, m_currentcolumn);
	else	{
		do	{
			pDoc->deletechar(m_currentrow, m_currentcolumn, i == m_currentcolumn);
			i++;
		} while (i < len && (isalpha(line[i]) || line[i] == '_'));
		while (i < len && isspace(line[i]))	{
			pDoc->deletechar(m_currentrow, m_currentcolumn, false);
			i++;
		}
	}
	UpdateTextLine();
}

void CEditorView::DeleteCharacter()
{
	m_buildcomments = true;
	if (IsSelectionMarked())	{
		DeleteSelection();
		return;
	}

	CEditorDoc* pDoc = GetDocument();
	ASSERT(pDoc != 0);
	
	bool blankline = pDoc->textline(m_currentrow).length() == 0 && 
				pDoc->linecount() < m_windowrows / m_fontheight;

	if (pDoc->deletechar(m_currentrow, m_currentcolumn) && !blankline)
		UpdateTextLine();
	else
		Invalidate(false);
}
void CEditorView::SplitLine()
{
	m_buildcomments = true;
	bool bsel = IsSelectionMarked();
	if (bsel)
		DeleteSelection();
	// --- Enter key, split the current string in two
	GetDocument()->insertchar('\r', m_currentrow, m_currentcolumn, !bsel);
	HScrollTo(0);
	Invalidate(false);
}

void CEditorView::InsertTab()
{
	m_buildcomments = true;
	CEditorDoc* pDoc = GetDocument();
	ASSERT(pDoc != 0);
	int tabs = theApp.Tabstops();
	std::string blankstr(tabs, ' ');
	bool bt = theApp.TabOption() == 0;

	if (IsSelectionMarked())	{
		ViewIndex begin, end;
		m_pEditorSelection->GetSelectionMarkers(begin, end);
		if (end.column > 0)
			end.row++;
		m_pEditorSelection->SaveSelection();
		m_pEditorSelection->UnmarkSelection();
		m_currentcolumn = 0;
		bool bterminate = true;
		if (m_shiftdown)	{
			for (m_currentrow = begin.row; m_currentrow < end.row; m_currentrow++)	{
				if (pDoc->textline(m_currentrow)[0] == '\t')	{
					pDoc->deletechar(m_currentrow, 0, bterminate);
					bterminate = false;
				}
				else	{
					int t = tabs;
					if (pDoc->textline(m_currentrow).substr(0,tabs) == blankstr)	{
						while (t--)	{
							pDoc->deletechar(m_currentrow, 0, bterminate);
							bterminate = false;
						}
					}
				}
				UpdateTextLine();
			}
		}
		else	{
			for (m_currentrow = begin.row; m_currentrow < end.row; m_currentrow++)	{
				if (bt)	{
					pDoc->insertchar('\t', m_currentrow, 0, bterminate);
					bterminate = false;
				}
				else	{
					int t = tabs;
					while (t--)	{
						pDoc->insertchar(' ', m_currentrow, 0, bterminate);
						bterminate = false;
					}
				}
				UpdateTextLine();
			}
		}
		m_pEditorSelection->RestoreSelection();
		UpdateWindow();
	}
	else	{
		pDoc->insertchar(bt ? '\t' : ' ', m_currentrow, m_currentcolumn);
		while (++m_currentcolumn % tabs)
			if (!bt)
				pDoc->insertchar(' ', m_currentrow, m_currentcolumn);
		UpdateTextLine();
		AdjustScroll();
	}
	SetCaretPosition(false, true);
}
void CEditorView::InsertCharacter(char ch, bool bterminate)
{
	m_buildcomments = true;
	bool bsel = IsSelectionMarked();
	if (bsel)
		DeleteSelection();
	GetDocument()->insertchar(ch, m_currentrow, m_currentcolumn, (!bsel) && bterminate);
	if (ch == '/' || ch == '*' || ch == '"')
		Invalidate(false);
	else
		UpdateTextLine();
}
void CEditorView::UpdateTextLine()
{
	m_buildcomments = true;
	if ((m_selectionmarked = IsSelectionMarked()) == true)
		m_pEditorSelection->GetSelectionMarkers(m_start, m_stop);
	CClientDC dc(this);
	delete m_pClearBrush;
	m_pClearBrush = new CBrush(theApp.BackgroundColor());
	InsertLineIntoScreen(&dc, m_currentrow);
	delete m_pClearBrush;
	m_pClearBrush = 0;
}

void CEditorView::DeleteSelection(bool changecaret, bool bterminate)
{
	m_buildcomments = true;
	ViewIndex begin, end;
	m_pEditorSelection->GetSelectionMarkers(begin, end);
	m_pEditorSelection->UnmarkSelection();
	const std::string& str = GetDocument()->textline(begin.row);
	std::string newstr = str.substr(0, begin.column);
	if (begin.row == end.row)
		// --- the selection is all on one line
		newstr += str.substr(end.column, str.length() - end.column);
	else	{
		while (begin.row != end.row)	{
			GetDocument()->deleteline(begin.row, bterminate);
			if (!changecaret && begin.row < m_currentrow)
				--m_currentrow;
			bterminate = false;
			--end.row;
		}
		if (begin.row < GetDocument()->linecount() - 1)	{
			const std::string& str = GetDocument()->textline(begin.row);
			newstr += str.substr(end.column, str.length() - end.column);
		}
	}
	GetDocument()->replaceline(end.row, newstr, bterminate);
	Invalidate(false);

	if (changecaret)	{
		m_currentrow = begin.row;
		m_currentcolumn = begin.column;
	}
	else if (end.row == m_currentrow && end.column < m_currentcolumn)
		m_currentcolumn -= m_currentcolumn - end.column;
	SetCaretPosition();
}
/////////////////////////////////////////////////////////////////////////////
// marked selection functions
void CEditorView::InvertDisplayItems(ViewIndex begin, ViewIndex end)
{
	CRect rc;
	rc.top = begin.row * m_fontheight;
	rc.bottom = end.row * m_fontheight + m_fontheight;
	rc.left = 0;
	rc.right = m_windowcolumns;
	do	{
		int len;
		if (begin.row < end.row)	{
			len = GetDocument()->textline(begin.row).length();
			// --- if its an empty line, make it look like one space character
			if (len == 0)
				len = 1;
		}
		else
			len = end.column;
		InvertDisplayRow(begin.row, begin.column, len);
		begin.column = 0;
	} while (begin.row++ < end.row);
}

void CEditorView::InvertDisplayRow(unsigned int row, unsigned int begincol, unsigned int endcol)
{
	CRect rc;
	rc.left = begincol * m_fontwidth - m_leftcolumn + m_margin;
	rc.top = row * m_fontheight - m_toprow;
	rc.right = endcol * m_fontwidth - m_leftcolumn + m_margin;
	rc.bottom = rc.top + m_fontheight;
	CClientDC dc(this);
	dc.PatBlt(rc.left,rc.top,rc.Width(),rc.Height(),DSTINVERT);
}

/////////////////////////////////////////////////////////////////////////////
// CEditorView message handlers
void CEditorView::OnVScroll(UINT nSBCode, UINT nPos, CScrollBar* pScrollBar) 
{
	SCROLLINFO sivert = {sizeof(SCROLLINFO)};
	GetScrollInfo(SB_VERT, &sivert);
	nPos = sivert.nTrackPos;
	switch(nSBCode)	{
		case SB_TOP:
			VScrollTo(0);
			break;
		case SB_THUMBTRACK:
			VScrollTo(nPos);
			break;
		case SB_LINEDOWN:
			ScrollUp();
			break;
		case SB_LINEUP:
			ScrollDown();
			break;
		case SB_PAGEDOWN:
			PageDown();
			break;
		case SB_PAGEUP:
			PageUp();
			break;
		default:
			break;
	}
}

void CEditorView::OnHScroll(UINT nSBCode, UINT nPos, CScrollBar* pScrollBar) 
{
	SCROLLINFO sihorz = {sizeof(SCROLLINFO)};
	GetScrollInfo(SB_HORZ, &sihorz);
	nPos = sihorz.nTrackPos;
	switch(nSBCode)	{
		case SB_LEFT:
			HScrollTo(0);
			break;
		case SB_LINELEFT:
			ScrollLeft();
			break;
		case SB_LINERIGHT:
			ScrollRight();
			break;
		case SB_PAGELEFT:
			PageLeft();
			break;
		case SB_PAGERIGHT:
			PageRight();
			break;
		case SB_THUMBTRACK:
			HScrollTo(nPos);
			break;
		default:
			break;
	}
}

// ------ convert screen coordinates to document character coordinates, 
//        adjusting for scroll position, and line lengths
CPoint CEditorView::ScreenToCharCoordinates(CPoint pt)
{
	if (pt.x < m_margin)
		pt.x = m_margin;
	int x = ((pt.x - m_margin) + m_leftcolumn) / m_fontwidth;
	int y = (pt.y + m_toprow) / m_fontheight;
	if (y >= GetDocument()->linecount())	{
		y = GetDocument()->linecount() - 1;
		if (y < 0)
			y = 0;
	}
	int wd = GetDocument()->textline(y).length();
	if (x > wd)
		x = wd;
	return CPoint(x, y);
}

// ------ convert document character coordinates to screen coordinates, 
CPoint CEditorView::CharToScreenCoordinates(CPoint pt)
{
	return CPoint(pt.x * m_fontwidth - m_leftcolumn + m_margin, pt.y * m_fontheight - m_toprow);
}

// ---- set the caret position adjusting for tabs in the text
void CEditorView::SetCaretPosition(bool bmoveright, bool frompaging)
{
	if (m_currentrow < 0)
		m_currentrow = 0;
	if (m_currentcolumn < 0)
		m_currentcolumn = 0;
	assert(GetDocument()->linecount() > 0);
	if (m_currentrow < GetDocument()->linecount())	{
		const std::string& str = GetDocument()->textline(m_currentrow);
		if (!str.empty())	{
			int i = m_currentcolumn;
			int tabs = theApp.Tabstops();
			while ((i % tabs) != 0 && str[i] == ' ')
				if (str[--i] == '\t')
					break;
			if (str[i] == '\t')	{
				if (i != m_currentcolumn && bmoveright)	{
					do
						i++;
					while ((i % tabs) != 0);
				}
				m_currentcolumn = i;
			}
		}
		if (m_currentcolumn > str.length())
			m_currentcolumn = str.length();
	}
	else	{
		m_currentrow = GetDocument()->linecount() - 1;
		m_currentcolumn = GetDocument()->textline(m_currentrow).length();
	}
	SetCaretPos(CharToScreenCoordinates(CPoint(m_currentcolumn, m_currentrow)));

	if (frompaging && m_pEditorSelection->IsMarking())
		m_pEditorSelection->ExtendSelection(ViewIndex(this, m_currentrow, m_currentcolumn));
	((CMainFrame*)theApp.m_pMainWnd)->SetRowColumn(m_currentrow+1, m_currentcolumn+1);
	ShowCaret();
}

void CEditorView::OnSetFocus(CWnd* pOldWnd) 
{
	CView::OnSetFocus(pOldWnd);
	CreateSolidCaret(2, m_fontheight);
	if (GetDocument()->linecount() > 0)
		SetCaretPosition();
}

void CEditorView::OnKillFocus(CWnd* pNewWnd) 
{
	CView::OnKillFocus(pNewWnd);
	((CMainFrame*)theApp.m_pMainWnd)->SetRowColumn(-1, -1);
	HideCaret();
	::DestroyCaret();
}

////////////////////////////////////////////////////////
// mouse functions

void CEditorView::OnLButtonDblClk(UINT nFlags, CPoint point) 
{
	if (!m_recording)	{
		m_pEditorSelection->UnmarkSelection();
		const std::string* str = &GetDocument()->textline(m_currentrow);
		if (!str->empty())	{
			if (iswordchar((*str)[m_currentcolumn]))	{
				while (m_currentcolumn > 0 && iswordchar((*str)[m_currentcolumn]))
					CaretLeft();
				if (!iswordchar((*str)[m_currentcolumn]))
					CaretRight();
				m_pEditorSelection->SetMarking(ViewIndex(this, m_currentrow, m_currentcolumn));
				do
					CaretRight();
				while (iswordchar((*str)[m_currentcolumn]));
				m_pEditorSelection->ExtendSelection(ViewIndex(this, m_currentrow, m_currentcolumn));
			}
		}
	}
	CView::OnLButtonDblClk(nFlags, point);
}

void CEditorView::OnLButtonDown(UINT nFlags, CPoint point) 
{
	if (!m_recording)	{
		CPoint pt = ScreenToCharCoordinates(point);
		m_currentrow = pt.y;
		m_currentcolumn = pt.x;
		SetCaretPosition();

		ReleaseCapture();

		if (m_pEditorSelection->IsItemInSelection(ViewIndex(this, m_currentrow, m_currentcolumn)))	{
			SetCapture();
			m_hOldCursor = SetCursor(m_ctrldown ? m_hCopyCursor : m_hMoveCursor);
			m_dragging = true;
			m_pEditorSelection->SaveSelection();
			return;
		}

		m_pEditorSelection->UnmarkSelection();

		m_mousemarking = true;
		m_pEditorSelection->SetMarking(ViewIndex(this, m_currentrow, m_currentcolumn));
		KillTimer(0);
		SetCapture();
	}
	CView::OnLButtonDown(nFlags, point);
}

void CEditorView::OnLButtonUp(UINT nFlags, CPoint point) 
{
	KillTimer(0);
	if (!m_recording)	{
		if (m_dragging)	{
			ViewIndex cvndx(this, m_currentrow, m_currentcolumn);
			if (!m_pEditorSelection->IsItemInSelection(cvndx))	{
				// ---- build a vector from the selection
				std::string str;
				SelectionToString(str);
				std::vector<std::string> strs;
				StringToVector(strs, str.c_str());

				if (!m_ctrldown)
					// ---- moving, delete the marked selection shifting instead of resetting the caret
					DeleteSelection(false);

				// ---- drop the marked selection into the document
				InsertStringsIntoDocument(strs, false);

				if (m_ctrldown)
					// ---- copying, reinvert the marked selection display
					m_pEditorSelection->RestoreSelection();
				Invalidate(false);
				SetCursor(m_hOldCursor);
			}
			else
				m_pEditorSelection->UnmarkSelection();
			m_dragging = false;
			ReleaseCapture();
		}
		else if (m_mousemarking)	{
			m_mousemarking = false;
			m_pEditorSelection->StopMarking();
			ReleaseCapture();
		}
	}
	CView::OnLButtonUp(nFlags, point);
}

void CEditorView::OnMouseMove(UINT nFlags, CPoint point) 
{
	if (!m_recording && (m_mousemarking || m_dragging))	{
		m_ScrollPos = point;
		if (point.y < 0)	{
			CaretUp();
			SetTimer(0, 100, 0);
			return;
		}
		if (point.y > m_windowrows)	{
			CaretDown();
			SetTimer(0, 100, 0);
			return;
		}
		CPoint pt = ScreenToCharCoordinates(point);
		if (pt.x >= 0 && pt.x < m_windowcolumns)	{
			m_currentrow = pt.y;
			m_currentcolumn = pt.x;
			SetCaretPosition();
			if (m_mousemarking)
				m_pEditorSelection->ExtendSelection(ViewIndex(this, m_currentrow, m_currentcolumn));
		}
	}
}

void CEditorView::OnTimer(UINT nIDEvent) 
{
	if (nIDEvent == 0)	{
		KillTimer(0);
		OnMouseMove(0, m_ScrollPos);
	}
	CView::OnTimer(nIDEvent);
}

////////////////////////////////////////////////////
// keyboard functions

void CEditorView::TestSelection()
{
	if (!m_shiftdown && IsSelectionMarked())
		m_pEditorSelection->UnmarkSelection();
}
void CEditorView::TestMarking()
{
	if (m_shiftdown)
		m_pEditorSelection->SetMarking(ViewIndex(this, m_currentrow, m_currentcolumn));
}

void CEditorView::OnKeyDown(UINT nChar, UINT nRepCnt, UINT nFlags) 
{
	if (m_recording)
		RecordCommand(new KeyDown(this, nChar));
	if (ExtendEditor(nChar))
		return;
	switch (nChar)	{
		case VK_TAB:
			InsertTab();
			return;
		case VK_BACK:
			if (m_currentrow || m_currentcolumn)	{
				CaretLeft();
				DeleteCharacter();
			}
			return;
		case VK_RETURN:
			SplitLine();
			CaretRight();
			return;
		case VK_DELETE:
			if (m_ctrldown)
				DeleteWord();
			else
				DeleteCharacter();
			return;
		default:
			break;
	}

	switch (nChar)	{
		case VK_CONTROL:
			m_ctrldown = true;
			return;
		case VK_SHIFT:
			m_shiftdown = true;
			return;
		case VK_DOWN:
			TestSelection();
			if (m_ctrldown)
				ScrollUp();
			else	{
				TestMarking();
				CaretDown();
			}
			return;
		case VK_UP:
			TestSelection();
			if (m_ctrldown)
				ScrollDown();
			else	{
				TestMarking();
				CaretUp();
			}
			return;
		case VK_RIGHT:
			TestSelection();
			TestMarking();
			m_ctrldown ? WordRight() : CaretRight();
			return;
		case VK_LEFT:
			TestSelection();
			TestMarking();
			m_ctrldown ? WordLeft() : CaretLeft();
			return;
		case VK_NEXT:
			TestSelection();
			if (!m_ctrldown)	{
				TestMarking();
				PageDown();
				m_currentrow += m_windowrows / m_fontheight;
				SetCaretPosition(false, true);
			}
			return;
		case VK_PRIOR:
			TestSelection();
			if (!m_ctrldown){
				TestMarking();
				PageUp();
				m_currentrow -= m_windowrows / m_fontheight;
				SetCaretPosition(false, true);
			}
			return;
		case VK_END:
			TestSelection();
			TestMarking();
			if (m_ctrldown)
				PageEnd();
			else
				LineEnd();
			return;
		case VK_HOME:
			TestSelection();
			TestMarking();
			if (m_ctrldown)
				PageHome();
			else
				LineHome();
			return;
		default:
			break;
	}
}

void CEditorView::OnKeyUp(UINT nChar, UINT nRepCnt, UINT nFlags) 
{
	if (m_recording && (nChar == VK_CONTROL || nChar == VK_SHIFT))
		RecordCommand(new KeyUp(this, nChar));
	if (nChar == VK_CONTROL)	
		m_ctrldown = false;
	else if (nChar == VK_SHIFT)
		m_shiftdown = false;
}

void CEditorView::OnChar(UINT nChar, UINT nRepCnt, UINT nFlags) 
{
	if (isprint(nChar))	{
		if (m_recording)
			RecordCommand(new Char(this, nChar));
		if (!ExtendEditor(nChar))	{
			InsertCharacter(nChar);
			CaretRight();
		}
	}
}

////////////////////////////////////////////////////////
// macro functions

void CEditorView::DeleteMacros()
{
	std::vector<MacroCommand*>::iterator iter = m_macrocommands.begin();
	while (iter != m_macrocommands.end())
		delete *(iter++);
	m_macrocommands.clear();
}

void CEditorView::OnRecord() 
{
	m_recording = true;
	CMenu* pMenu = ((CMainFrame*)theApp.m_pMainWnd)->GetMenu();
	assert(pMenu != 0);
	pMenu->ModifyMenu(ID_RECORD, MF_BYCOMMAND | MF_STRING, ID_STOPRECORD, "&Stop\tAlt+S");
	DeleteMacros();
	SetCursor(m_hRecordCursor);
}

void CEditorView::OnStopRecord() 
{
	m_recording = false;
	CMenu* pMenu = ((CMainFrame*)theApp.m_pMainWnd)->GetMenu();
	assert(pMenu != 0);
	pMenu->ModifyMenu(ID_STOPRECORD, MF_BYCOMMAND | MF_STRING, ID_RECORD, "&Record\tAlt+R");
	SetCursor(m_hArrowCursor);
}

void CEditorView::RecordCommand(MacroCommand* pCmd)
{
	m_macrocommands.push_back(pCmd);
}

void CEditorView::OnPlayback() 
{
	std::vector<MacroCommand*>::iterator iter = m_macrocommands.begin();
	while (iter != m_macrocommands.end())	{
		(*iter)->PlayBack();
		iter++;
	}
}

void CEditorView::OnUpdatePlayback(CCmdUI* pCmdUI) 
{
	pCmdUI->Enable(!m_recording && !m_macrocommands.empty());
}

void CEditorView::OnUpdateRecord(CCmdUI* pCmdUI) 
{
	pCmdUI->Enable(!m_recording);
}

void CEditorView::OnUpdateStopRecord(CCmdUI* pCmdUI) 
{
	pCmdUI->Enable(m_recording);
}

////////////////////////////////////////////////////////
// clipboard functions


void CEditorView::OnUpdateEditCut(CCmdUI* pCmdUI) 
{
	pCmdUI->Enable(IsSelectionMarked());
}

//void CEditorView::OnUpdateEditCopy(CCmdUI* pCmdUI) 
//{
//	pCmdUI->Enable(IsSelectionMarked());
//}

void CEditorView::OnUpdateEditPaste(CCmdUI* pCmdUI) 
{
	pCmdUI->Enable(IsClipboardFormatAvailable(CF_TEXT));
}

void CEditorView::OnUpdateEditClear(CCmdUI* pCmdUI) 
{
	pCmdUI->Enable(!IsRecording() && IsSelectionMarked());
}

void CEditorView::OnEditCut() 
{
	OnEditCopy();
	OnEditClear();
}

void CEditorView::OnEditClear() 
{
	DeleteSelection();
}

void CEditorView::OnEditSelectAll() 
{
	m_pEditorSelection->UnmarkSelection();
	PageHome();
	m_pEditorSelection->SetMarking(ViewIndex(this, 0, 0));
	int endrow = GetDocument()->linecount() - 1;
	int endcol = GetDocument()->textline(endrow).length();
	m_pEditorSelection->ExtendSelection(ViewIndex(this, endrow, endcol));
}
// ------ functions to support view contexts for undo/redo
void CEditorView::GetCurrentPosition(unsigned int& toprow, unsigned int& leftcolumn, unsigned int& row, unsigned int& column) const
{
	toprow = m_toprow;
	leftcolumn = m_leftcolumn;
	row = m_currentrow;
	column = m_currentcolumn;
}
void CEditorView::SetCurrentPosition(unsigned int toprow, unsigned int leftcolumn, unsigned int row, unsigned int column)
{
	if (m_leftcolumn != leftcolumn || m_toprow != toprow)
		Invalidate(false);	// changing scroll position
	m_toprow = toprow;
	m_leftcolumn = leftcolumn;
	m_currentrow = row;
	m_currentcolumn = column;
	SetScrollPosition(SB_VERT, m_toprow);
	SetCaretPosition();
	AdjustScroll();
}

// --- copy the text selection to one string, collapsing tab expansions
//     and inserting \r\n between lines
//     Used by cut, copy and drop operations
void CEditorView::SelectionToString(std::string& str)
{
	ViewIndex begin, end;
	m_pEditorSelection->GetSelectionMarkers(begin, end);
	std::string strline = GetDocument()->textline(begin.row);
	if (begin.row == end.row)	{
		// --- the selection is all on one line
		str = strline.substr(begin.column, end.column - begin.column);
		CEditorDoc::entab(str);
	}
	else	{
		str = strline.substr(begin.column, strline.length() - begin.column) + "\r\n";
		CEditorDoc::entab(str);
		while (++begin.row != end.row)	{
			strline = GetDocument()->textline(begin.row);
			CEditorDoc::entab(strline);
			str += strline + "\r\n";
		}
		if (end.row != GetDocument()->linecount())	{
			strline = GetDocument()->textline(end.row);
			CEditorDoc::entab(strline);
			str += strline.substr(0, end.column);
			if (end.column == 0)
				str += "\r\n";
		}
	}
}

// --- build a vector of strings from a null-terminated string with "\r\n" separators
//     Used by paste and drop operations
void CEditorView::StringToVector(std::vector<std::string>& strs, const char* cp)
{
	std::string str;
	while (*cp)	{
		str += *cp;
		if (*cp == '\n')	{
			strs.push_back(str.substr(0, str.length()-2));
			str.erase();
		}
		cp++;
	}
	if (str.length() > 0)
		strs.push_back(str);
}

// --- insert a vector of text line strings into the document at the current insertion point,
//     expanding tabs
//     Used by paste and drop operations
void CEditorView::InsertStringsIntoDocument(const std::vector<std::string>& strs, bool bterminate)
{
	if (strs.size() > 0)	{
		m_buildcomments = true;
		const std::string& str = GetDocument()->textline(m_currentrow);
		std::string first = str.substr(0, m_currentcolumn);
		std::string last = str.substr(m_currentcolumn, str.length() - m_currentcolumn);
		std::string mid = *strs.begin();

		if (strs.size() == 1)	{
			// --- only one text line in the vector
			int ccol = first.length();
			CEditorDoc::detab(mid);
			ccol += mid.length();
			CEditorDoc::entab(mid);
			CEditorDoc::entab(first);
			CEditorDoc::entab(last);
			std::string str(first+mid+last);
			CEditorDoc::detab(str);
			GetDocument()->replaceline(m_currentrow, str, bterminate);
			m_currentcolumn = ccol;
		}
		else	{
			// ---- process the first line from the vector
			CEditorDoc::entab(first);
			std::string str(first+mid);
			CEditorDoc::detab(str);
			GetDocument()->replaceline(m_currentrow, str, bterminate);

			int inserter = m_currentrow + 1;

			// ---- process the middle lines from the vector
			if (strs.size() > 2)	{
				std::vector<std::string>::const_iterator iter;
				for (iter = strs.begin() + 1; iter != strs.end()-1; iter++)	{
					std::string mstr = *iter;
					CEditorDoc::detab(mstr);
					GetDocument()->insertline(inserter++, mstr, false);
				}
			}
			// ---- process the last line from the vector
			std::string lstr = *(strs.end()-1);
			CEditorDoc::detab(lstr);
			unsigned int ccol = lstr.length();
			CEditorDoc::entab(lstr);
			str = lstr + last;
			CEditorDoc::detab(str);
			if (inserter >= GetDocument()->linecount())
				GetDocument()->appendline(str, false);
			else
				GetDocument()->insertline(inserter, str, false);
			m_currentcolumn = ccol;
			m_currentrow = inserter;
		}
	}
}

void CEditorView::StringToClipboard(const std::string& str)
{
	HGLOBAL hGlobalMemory = GlobalAlloc(GHND, str.length() + 1);
	if (hGlobalMemory != 0)	{
		char* pGlobalMemory = (char*) GlobalLock(hGlobalMemory);
		if (pGlobalMemory != 0)	{
			int len = str.length();
			const char* cp = str.c_str();
			while (len--)
				*pGlobalMemory++ = *cp++;
		}
		GlobalUnlock(hGlobalMemory);
		OpenClipboard();
		EmptyClipboard();
		SetClipboardData(CF_TEXT, hGlobalMemory);
		CloseClipboard();
	}
}

void CEditorView::OnEditCopy() 
{
	std::string str;
	SelectionToString(str);
	if (str.empty())
		str = GetDocument()->textline(m_currentrow);
	StringToClipboard(str);
}

void CEditorView::OnEditPaste() 
{
	OpenClipboard();
	HANDLE hClipboardData = GetClipboardData(CF_TEXT);
	std::vector<std::string> strs;
	bool bsel = m_pEditorSelection->IsSelectionMarked();
	if (hClipboardData != 0)	{
		char* cp = (char*) GlobalLock(hClipboardData);
		if (cp != 0)	{
			if (bsel)
				DeleteSelection();
			// --- build a vector of strings from the clipboard \r\n terminated strings
			StringToVector(strs, cp);
		}
		GlobalUnlock(hClipboardData);
	}
	CloseClipboard();
	if (strs.size() > 0)	{
		InsertStringsIntoDocument(strs, !bsel);
		Invalidate(false);
		SetCaretPosition();
	}
}

////////////////////////////////////////////////////////
// search and replace functions

void CEditorView::OnEditFind() 
{
	m_ctrldown = false;			// in case of Ctrl+F
	FindReplace(true);
}

void CEditorView::OnEditReplace() 
{
	m_ctrldown = false;			// in case of Ctrl+H
	FindReplace(false);
}

void CEditorView::OnUpdateEditFindNext(CCmdUI* pCmdUI) 
{
	pCmdUI->Enable(m_findtext.length() != 0);
}

void CEditorView::OnEditFindNext() 
{
	FindNext();
}

void CEditorView::FindReplace(bool findonly) 
{
	std::string str = SelectedText();
	if (!str.empty())
		m_findtext = str;
	m_pEditorSelection->UnmarkSelection();
	m_wasfound = m_found = m_bchanged = false;
	delete m_pFindDlg;
	m_pFindDlg = new CFindReplaceDialog;
	m_pFindDlg->Create(findonly, m_findtext.c_str(), m_replacetext.empty() ? 0 : m_replacetext.c_str(), m_nfrflags, this);
}

void CEditorView::FindNext()
{
	m_pEditorSelection->UnmarkSelection();
	if (m_nfrflags & FR_DOWN)
		FindTextForward();
	else
		FindTextBackward();
	if (m_found)	{
		int len = m_findtext.length();
		m_currentrow = m_findrow;
		m_currentcolumn = m_findcol;
		if (m_nfrflags & FR_DOWN)
			m_currentcolumn += len;
		else
			m_findcol += len;
		m_pEditorSelection->SetMarking(ViewIndex(this, m_findrow, m_findcol));
		m_pEditorSelection->ExtendSelection(ViewIndex(this, m_currentrow, m_currentcolumn));
		AdjustScrollQuietly();
	}
}

LONG CEditorView::OnFindReplace(WPARAM wParam, LPARAM lParam)
{
	LPFINDREPLACE fr = (LPFINDREPLACE)lParam;
	m_findtext = fr->lpstrFindWhat;
	m_replacetext = fr->lpstrReplaceWith;
	m_nfrflags = fr->Flags;

	if (m_nfrflags & FR_DIALOGTERM)	{
		m_pFindDlg = 0;
		m_wasfound = m_found = m_bchanged = false;
		m_nfrflags &= ~FR_DIALOGTERM;
		return 0;
	}
	
	int saverow = m_currentrow;
	int savecol = m_currentcolumn;

	if (m_nfrflags & FR_REPLACEALL)
		m_currentrow = m_currentcolumn = 0;
	bool bterminate = true;
	for (;;)	{
		if (((m_nfrflags & FR_REPLACE) || (m_nfrflags & FR_REPLACEALL)) && m_found)	{
			assert(m_pEditorSelection->IsSelectionMarked());
			ViewIndex start, stop;
			m_pEditorSelection->GetSelectionMarkers(start, stop);
			m_pEditorSelection->UnmarkSelection();
			assert(start.row == stop.row);
			assert(start.column < stop.column);
			const std::string& line = GetDocument()->textline(start.row);
			std::string first = line.substr(0, start.column);
			std::string last = line.substr(stop.column, line.length() - stop.column);
			std::string mid(m_replacetext);
			std::string line1 = first + mid + last;
			GetDocument()->replaceline(start.row, line1, bterminate);
			m_currentrow = start.row;
			m_currentcolumn = start.column + m_replacetext.length();
			bterminate = false;
			m_bchanged = true;
			m_buildcomments = true;
		}
		else
			m_pEditorSelection->UnmarkSelection();
		int len = m_findtext.length();
		FindNext();
		m_wasfound |= m_found;
		if (!m_found)	{
			if (!m_wasfound)	{
				CString msg("Cannot find \"");
				msg += m_findtext.c_str();
				msg += '"';
				AfxMessageBox(msg, MB_ICONEXCLAMATION);
				m_currentrow = saverow;
				m_currentcolumn = savecol;
			}
			break;
		}
		if (!(m_nfrflags & FR_REPLACEALL))
			break;
	}
	if (m_bchanged)
		AdjustScrollQuietly();
	return 0;
}

bool CEditorView::isword(const char* cp, int col, int len)
{
	if (col == 0 || isspace(cp[col-1]) || ispunct(cp[col-1]))
		if (cp[col + len] == '\0' || isspace(cp[col + len]) || ispunct(cp[col + len]))
			return true;
	return false;
}

void CEditorView::FindTextForward()
{
	m_found = false;
	m_findrow = m_currentrow;
	m_findcol = m_currentcolumn;
	int len = m_findtext.length();
	while (m_findrow < GetDocument()->linecount())	{
		const std::string& line = GetDocument()->textline(m_findrow);
		while (m_findcol + len <= line.length())	{
			bool whole = (m_nfrflags & FR_WHOLEWORD) != 0;
			if ((m_nfrflags & FR_MATCHCASE) == 0)	{
				if (strnicmp(m_findtext.c_str(), line.c_str() + m_findcol, len) == 0)	{
					if (!whole || isword(line.c_str(), m_findcol, len))	{
						m_found = true;
						return;
					}
				}
			}
			else if (strncmp(m_findtext.c_str(), line.c_str() + m_findcol, len) == 0)	{
				if (!whole || isword(line.c_str(), m_findcol, len))	{
					m_found = true;
					return;
				}
			}
			m_findcol++;
		}
		m_findcol = 0;
		m_findrow++;
	}
}

void CEditorView::FindTextBackward()
{
	m_found = false;
	m_findrow = m_currentrow;
	m_findcol = m_currentcolumn;
	int len = m_findtext.length();
	while (m_findrow >= 0)	{
		const std::string& line = GetDocument()->textline(m_findrow);
		m_findcol -= len;
		while (m_findcol >= 0)	{
			bool whole = (m_nfrflags & FR_WHOLEWORD) != 0;
			if ((m_nfrflags & FR_MATCHCASE) == 0)	{
				if (strnicmp(m_findtext.c_str(), line.c_str() + m_findcol, len) == 0)	{
					if (!whole || isword(line.c_str(), m_findcol, len))	{
						m_found = true;
						return;
					}
				}
			}
			else if (strncmp(m_findtext.c_str(), line.c_str() + m_findcol, len) == 0)	{
				if (!whole || isword(line.c_str(), m_findcol, len))	{
					m_found = true;
					return;
				}
			}
			--m_findcol;
		}
		if (--m_findrow >= 0)
			m_findcol = GetDocument()->textline(m_findrow).length();
	}
}

////////////////////////////////////////////////////////
// undo/redo functions

void CEditorView::OnEditUndo() 
{
	m_buildcomments = true;
	m_pEditorSelection->UnmarkSelection();
	GetDocument()->OnEditUndo();
}

void CEditorView::OnEditRedo() 
{
	m_buildcomments = true;
	m_pEditorSelection->UnmarkSelection();
	GetDocument()->OnEditRedo();
}

BOOL CEditorView::OnSetCursor(CWnd* pWnd, UINT nHitTest, UINT message) 
{
	SetCursor(m_recording ? m_hRecordCursor : m_hArrowCursor);
	return true;
}

void CEditorView::OnFileOpen() 
{
	m_buildcomments = true;
	m_ctrldown = false;			// in case of Ctrl+O
	theApp.OnFileOpen();
}

void CEditorView::OnFilePrint() 
{
	m_ctrldown = false;			// in case of Ctrl+P
	CView::OnFilePrint();
}

void CEditorView::OnUpdateFileSaveAll(CCmdUI* pCmdUI) 
{
	pCmdUI->Enable(true);
}

// ------- brace matching
void CEditorView::OnUpdateBracematch(CCmdUI* pCmdUI) 
{
	pCmdUI->Enable(IsSourceCodeFile() && CanMatchBraces());
}

void CEditorView::OnUpdateBeautify(CCmdUI* pCmdUI) 
{
	pCmdUI->Enable(pEditor == 0 && theApp.BeautifierIsInstalled() && IsSourceCodeFile());
}

// --- ensure that brace character is not character or string literal
//     and not in a comment
bool CEditorView::IsSpecificBrace(char ch, int row, int col)
{
	if (GetDocument()->textchar(row, col) != ch)
		return false;
	if (col > 0)	{
		if (IsCharLiteral(row, col))
			return false;
		if (IsContextElement(row, col))
			return false;
	}
	return true;
}

#ifdef _DEBUG
void CEditorView::DumpContextTable()
{
	std::set<CodeElement>::iterator iter;
	TRACE("\n -------- Context Table ----------");
	for (iter = m_contexttable.begin(); iter != m_contexttable.end(); iter++)	{
		CodeElement cmt = *iter;
		TRACE("\nLine %02d: Char %02d: Clr %06x", cmt.line, cmt.charpos, cmt.color);
	}
	TRACE("\n -------- End of Context Table ----------\n");
}
#endif

bool CEditorView::IsContextElement(int row, int col)
{
	CodeElement ce;
	ce.line = row;
	ce.charpos = col;

	std::set<CodeElement>::const_iterator it = std::lower_bound(m_contexttable.begin(), m_contexttable.end(), ce);
	if (it == m_contexttable.end())
		return false;
	if (it != m_contexttable.begin())
		--it;
	return (*it).color != theApp.NormalColor();
}

bool CEditorView::IsCharLiteral(int row, int col)
{
	const std::string& str = GetDocument()->textline(row);
	if (col < str.length()-1)	{
		if (GetDocument()->textchar(row, col-1) == '\'')
			if (GetDocument()->textchar(row, col+1) == '\'')
				return true;
	}
	return false;
}

void CEditorView::Beautify()
{
	PageHome();
	m_pEditorSelection->UnmarkSelection();
	beautifiedtext.clear();
	CString strExe = theApp.Enquote(theApp.QuincyBinPath() + "\\astyle.exe ");
	CString strArgs(theApp.BeautifierCommand());
	delete m_pConsoleApp;
	m_pConsoleApp = new ConsoleApp(strExe, &Notify, &Collect);
	try	{
		m_pConsoleApp->Run(strArgs);

		int lines = GetDocument()->linecount();
		if (lines != 0) {
			for (int i = 0; i < lines; i++) {
				std::string str = GetDocument()->textline(i);
				CEditorDoc::entab(str);
				m_pConsoleApp->WriteConsole(str.c_str());
//				TRACE("\nOUT: %s", str.c_str());
			}
		}
		char ctrlz[2] = {26,0};
		m_pConsoleApp->WriteConsole(ctrlz);
	}
	catch(...)	{
		AfxMessageBox("\nCannot execute Astyle", MB_ICONSTOP);
	}
}

// --- called from the console application thread while beautifying is going on
void CEditorView::Collect(DWORD bufct)
{
	ASSERT(pEditor != 0);
	pEditor->CollectSourceLines(bufct);
}

void CEditorView::CollectSourceLines(DWORD bufct)
{
	char *line = new char[bufct+1];
	ASSERT(m_pConsoleApp);
	if (m_pConsoleApp->ReadConsole(line, bufct) != 0)	{
//		TRACE("\nIN:  %s", line);
		std::string str(line);
		CEditorDoc::detab(str);
		beautifiedtext.push_back(str);
	}
	delete [] line;
}

void CEditorView::Notify()
{
	ASSERT(pEditor != 0);
	pEditor->NotifyTermination();
}

void CEditorView::NotifyTermination()
{
//	TRACE("\n---done---\n");
	beautifiedtext.push_back(std::string());
	GetDocument()->ReplaceDocument(beautifiedtext);
	beautifiedtext.clear();
	BuildContextHighlightTable();
	Invalidate(TRUE);
	delete m_pConsoleApp;
	m_pConsoleApp = 0;
	pEditor = 0;
}

void CEditorView::OnBeautify()
{
	CBeautifier bt;
	bt.m_style = theApp.BeautifierStyle();
	bt.m_Cmdline = theApp.BeautifierCommand();
	if (bt.DoModal() == IDOK)	{
		pEditor = this;
		theApp.SetBeautifierStyle(bt.m_style);
		theApp.SetBeautifierCommand(bt.m_Cmdline);
		Beautify();
	}
}

void CEditorView::OnBracematch()
{

#ifdef _DEBUG
//	DumpContextTable();
#endif

	// --- these are zero-based
	int row = m_currentrow;
	int col = m_currentcolumn;

	int lines = GetDocument()->linecount();

	if (IsLeftBrace(row, col))	{
		int nests = 0;
		bool bmatch = false;
		while (row < lines)	{
			int len = GetDocument()->textline(row).length();
			while (col < len)	{
				if (IsLeftBrace(row, col))
					nests++;
				else if (IsRightBrace(row, col))	{
					if ((bmatch = --nests == 0) == true)
						break;
				}
				if (bmatch)
					break;
				col++;
			}
			if (bmatch)
				break;
			row++;
			col = 0;
		}
		if (bmatch)	{
			m_currentrow = row;
			m_currentcolumn = col;
			AdjustScrollQuietly();
		}
		else	{
			AfxMessageBox("No matching '}'", MB_ICONSTOP);
			m_ctrldown = false;
		}
	}
	else if (IsRightBrace(row, col))	{
		int nests = 0;
		bool bmatch = false;
		while (row >= 0)	{
			while (col >= 0)	{
				if (IsRightBrace(row, col))
					nests++;
				else if (IsLeftBrace(row, col))	{
					if ((bmatch = --nests == 0) == true)
						break;
				}
				if (bmatch)
					break;
				--col;
			}
			if (bmatch)
				break;
			if (--row >= 0)
				col = GetDocument()->textline(row).length() - 1;
		}
		if (bmatch)	{
			m_currentrow = row;
			m_currentcolumn = col;
			AdjustScrollQuietly();
		}
		else
			AfxMessageBox("No matching '}'", MB_ICONSTOP);
	}
}


// ---- Editor calls this function for every character typed.
//      Use the CEditorView and CEditorDoc APIs to extend the editor.
//      return true if you don't want Editor to further process the keystroke
bool CEditorView::ExtendEditor(UINT nChar)
{
	if (theApp.AutoIndentOption() && nChar == VK_RETURN)	{
		AutoIndent();
		return true;
	}
	
	return false;
}
// --- automatic indenting puts insertion caret and any following text
//     under leftmost character of previous line after user presses Enter key
void CEditorView::AutoIndent()
{
	int lno = CurrentLineNumber(); // one-based line number
	ASSERT(lno > 0);
	SplitLine();
	CaretRight();

	// next line moved down from 2 statements up as it was returning a
	// reference that was invalidated in splitline, crashing on new files/
	const std::string& line = GetDocument()->textline(lno-1);  // zero-based to get current line
	int len = line.length();
	int i;
	for (i = 0; i < len; i++)
		if (line[i] != '\t' && line[i] != ' ')
			  break;
	int tbs = theApp.Tabstops();
	if (theApp.TabOption() == 0)	{
		while (i >= tbs)	{
			InsertCharacter('\t', false);
			i -= tbs;
			CaretRight();
		}
	}
	while (i-- > 0)	{
		InsertCharacter(' ', false);
		CaretRight();
	}
}

void CEditorView::OnActivateView(BOOL bActivate, CView* pActivateView, CView* pDeactiveView) 
{
	CView::OnActivateView(bActivate, pActivateView, pDeactiveView);
	theApp.RegisterView(pEditor);
}

void CEditorView::UpdateTitle(CDocument* pDoc)
{
	CString caption;
	CWnd* pWnd = GetParent();
	ASSERT(pWnd != 0);
	pWnd->GetWindowText(caption);
	if (!caption.IsEmpty())	{
		bool tagged = caption[caption.GetLength()-1] == '*';
		ASSERT(pDoc != 0);
		if (pDoc->IsModified())	{
			if (!tagged)
				pWnd->SetWindowText(caption + " *");
		}
		else if (tagged)
			pWnd->SetWindowText(caption.Left(caption.GetLength()-2));
	}
}

std::string CEditorView::SelectedText()
{
	std::string sstr;
	ASSERT(m_pEditorSelection);
	if (m_pEditorSelection->IsSelectionMarked())	{
		ViewIndex begin, end;
		m_pEditorSelection->GetSelectionMarkers(begin, end);
		if (begin.row == end.row)
			// --- the selection is all on one line
			SelectionToString(sstr);
	}
	else	{
		const std::string& str = GetDocument()->textline(m_currentrow);
		if (!str.empty())	{
			int col = m_currentcolumn;
			if (iswordchar(str[col]))	{
				while (col > 0 && iswordchar(str[col]))
					--col;
				if (!iswordchar(str[col]))
					col++;
				do
					sstr += str[col++];
				while (iswordchar(str[col]));
			}
		}
	}
	return sstr;
}

/*void CEditorView::OnUpdate(CView* pSender, LPARAM lHint,CObject* pHint)
{
		CSize size(GetDocument( )->linewidth( ), GetDocument( )->linecount( ) );
		SetScrollSizes( MM_LOENGLISH, size );
}*/
