// ----- compiler.h
#ifndef COMPILER_H
#define COMPILER_H

#include <queue>
#include "Quincy.h"

struct ConsoleCmd	{
	CString command;
	bool dependent;
};

class Compiler	{
	// ---- maximum number of errors reported 
	std::queue<ConsoleCmd, std::deque<ConsoleCmd> > cmds;
	static Compiler* pCompiler;
public:
	enum CompileAction { none, run, step, debug };
	ConsoleApp* m_pConsoleApp;
protected:
	std::vector<CString> MessageLog;
	PROCESS_INFORMATION m_CompileInformation;	// for compiling
	bool m_bLinkCpp;			// true if linking c++
	bool m_bCpp;				// true if compiling c++
	bool m_bFoundDef;			// true if def is in project
	bool m_bStopping;			// compiler being stopped by user

	CompileAction m_CompileAction;
	CStringArray m_SourceFiles;		// source files to compile
	CStringArray m_ObjectFiles;		// object files to link
	CStringArray m_LibraryFiles;	// library files to link
	CStringArray m_ResourceFiles;	// resource files to bind

	CString m_strTargetName;	// path of compiler file to build and/or execute
	CString m_strFilename;		// source code file name without path
	CString m_strDefname;		// .def file name
	CString m_strErrorFile;		// error stdout file from compiler

	CString m_strOFile;			// object file path and name
	CString m_strOPath;			// object file path
	CString m_strObjFiles;		// object file list
	CString m_strLibFiles;		// library file list
	DWORD ExitCode;
	CErrorLogDialog* m_pdlgErrors;	// error log dialog box
	bool m_bErrorCreated;			// true if error log list has been created
	int m_nErrorLogWidth;			// width of the widest message in the error log

private:
	static void Collect(DWORD bufct);
	void CollectErrorMessages(DWORD bufct);
	static void Notify();
	void NotifyTermination();
	void ScheduleProgram(CString& strCmd, bool depends = false);
protected:
	void RunCompilerProgram(const CString& strCmd);
	virtual void BuildResourceScriptCommand(CString& strCmd) = 0;
	virtual void BuildResFileCommand(CString& strCmd) = 0;
	virtual void BuildCompilerCommand(CString& strCmd, const CString& strFile) = 0;
	virtual void BuildLibCommand(CString& strCmd) = 0;
	virtual void BuildDLLCommand(CString& strCmd) = 0;
	virtual void BuildDefCommand(CString& strCmd, const CString& strFile) = 0;
	virtual void BuildLinkerCommand(CString& strCmd) = 0;
	virtual void CompileOneSource(const CString& strFile);

public:
	Compiler();
	~Compiler();
	bool CompileRunning() const;
	void Stop();
	void SetBuildingCPP(bool bSet)
		{ m_bLinkCpp = bSet; }
	void ClearArrays();
	void AddSourceFile(const CString& rstrSrc);
	void AddObjectFile(const CString& rstrObj);
	void AddResourceFile(const CString& rstrRc);
	virtual void CompileAllSources();
	virtual int  GatherObjects();
	virtual void BuildLibraryTarget();
	virtual void BuildExeTarget();
	virtual void BuildDLLTarget();
	virtual void GatherLibraries();
	virtual CString MakeObjectFileName(const CString& str) const = 0;
	virtual void AddLibraryFile(const CString& rstrLib) = 0;
	void BuildTarget(const CString& strTargetName, 
					CompileAction action = none);
	void LinkTarget(const CString& strTargetName);
	void CloseErrorLog();
	void ClearErrorLog();
	const CString& GetTargetName() const
		{ return m_strTargetName; }
	const CString& GetErrorFile() const
		{ return m_strErrorFile; }
	virtual bool GetMessageData(int sel, CString& strFile, int& line) = 0;
	void AddLogEntry(const CString& str);
};

class GCCCompiler : public Compiler	{
	CString m_strPrevLine;		// previous error message line
public:
	CString MakeObjectFileName(const CString& str) const
		{ return str + ".o"; }
	void CompileAllSources();
	void CompileOneSource(const CString& strFile);
	void AddLibraryFile(const CString& rstrLib);
	void BuildResourceScriptCommand(CString& strCmd);
	void BuildResFileCommand(CString& strCmd);
	void BuildCompilerCommand(CString& strCmd, const CString& strFile);
	void BuildLibCommand(CString& strCmd);
	void BuildDLLCommand(CString& strFile);
	void BuildDefCommand(CString& strCmd, const CString& strFile);
	void BuildLinkerCommand(CString& strCmd);
	void GetLibraryPaths(CString& strCmd);
	void GetGUILibraries(CString& strCmd);
	void GetFLTKLibraries(CString& strCmd);
	void GetKoolplotLibraries(CString& strCmd);
	void GetBGILibraries(CString& strCmd);
	bool GetMessageData(int sel, CString& strFile, int& line);
};

#endif
