// EditorView.h : interface of the CEditorView class
//
/////////////////////////////////////////////////////////////////////////////

#if !defined(AFX_EDITORVIEW_H__408454CF_548A_11D2_A779_00C0A80034F0__INCLUDED_)
#define AFX_EDITORVIEW_H__408454CF_548A_11D2_A779_00C0A80034F0__INCLUDED_

#if _MSC_VER >= 1000
#pragma once
#endif // _MSC_VER >= 1000

#include <vector>
#include <set>
#include "Selection.h"

class CEditorDoc;
class EditorSelection;
class CEditorView;
class MacroCommand;

//////////////////////////////////////////////
// ViewIndex 
//
// Supports marking selections

class ViewIndex	{
	unsigned int row, column;
	CEditorView* pView;
	friend CEditorView;
public:
	ViewIndex(CEditorView* pV = 0, unsigned int rw = 0, unsigned int col = 0) : pView(pV), row(rw), column(col) { }
	ViewIndex& operator=(const ViewIndex& view);
	bool operator==(const ViewIndex& view) const;
	bool operator<(const ViewIndex& view) const;
};

class CEditorView : public CView
{
	friend ViewIndex;
	std::vector<std::string> beautifiedtext;

	static std::string cppkeys[];
	static std::string ckeys[];
	std::set<std::string> keywords;

	CBrush* m_pClearBrush;
	CBrush* m_pMarginBrush;
	CPoint m_ScrollPos;

	HCURSOR m_hOldCursor;
	HCURSOR m_hMoveCursor;
	HCURSOR m_hCopyCursor;
	HCURSOR m_hRecordCursor;
	HCURSOR m_hArrowCursor;

	bool m_mousemarking;
	bool m_dragging;
	bool m_ctrldown;
	bool m_shiftdown;
	bool m_recording;

    CFont* m_pscreenfont;

    CFont m_printerfont;
    UINT m_cyPrinter;
    UINT m_nLinesPerPage;
	bool printlinenos;	// 

	EditorSelection* m_pEditorSelection;

	std::vector<MacroCommand*> m_macrocommands;

	// --- current insertion point in zero-based character coordinates
	int m_currentrow;
	int m_currentcolumn;
protected:
	unsigned int m_fontheight;
	unsigned int m_fontwidth;
	unsigned int m_fontweight;
	unsigned int m_margin;	// set to pixel value of left margin (for breakpoints, etc)
private:
	int m_leftcolumn;		// scroll positions in pixels
	int m_toprow;
	unsigned int m_bottomrow;
	unsigned int m_windowrows;
	unsigned int m_windowcolumns;
	unsigned int m_linewidth;

	// --- stuff for parsing out comments 
	int m_linecount;
	int m_linescount;
	int m_charcount;
	int m_linelength;
	std::string m_linestr;

	struct CodeElement	{
		int line, charpos;
		COLORREF color;
		bool operator<(const struct CodeElement& codeelement) const;
	};
	std::set<CodeElement> m_contexttable;
	bool m_inccomment;
	bool m_incppcomment;
	bool m_instringliteral;
	bool m_incharliteral;
	std::string kword;	// for collecting potential keywords
	COLORREF m_prevcolor;

	static CFindReplaceDialog* m_pFindDlg;
	static std::string m_findtext;
	static std::string m_replacetext;
	static int m_nfrflags;

	int m_findrow, m_findcol;
	bool m_found;
	bool m_wasfound;
	bool m_bchanged;

	bool m_buildcomments;
	bool m_selectionmarked;
	ViewIndex m_start, m_stop;


	static CEditorView* pEditor;
	ConsoleApp* m_pConsoleApp;
	static void Collect(DWORD bufct);
	void CollectSourceLines(DWORD bufct);
	static void Notify();
	void NotifyTermination();


    void PrintPageHeader(CDC*, UINT);
    void PrintPage(CDC*, UINT);

	bool IsSourceFile(const CString& ext);
	bool IsCFile()
		{ return IsSourceFile(".c"); }
	bool IsCPPFile()
		{ return    IsSourceFile(".cpp") 
				 || IsSourceFile(".cxx")
				 || IsSourceFile(".cc"); }
	bool IsHFile()
		{ return IsSourceFile(".h"); }
	bool IsRCFile()
		{ return IsSourceFile(".rc"); }
	bool IsSourceCodeFile()
		{ return IsCPPFile() || IsCFile() || IsHFile() || IsRCFile(); }

	void DisplayText(CDC* pDC, int beg, int len, int line, COLORREF color);
	void DisplaySourceText(CDC* pDC, int line);
	void DisplayCodeText(CDC* pDC,  int beg, int len, int line, COLORREF color);
	int GetNextSourceChar();

	void AddColorToTable(int line, int charpos, COLORREF clr);
	void KeywordSearch();

	void InsertLineIntoScreen(CDC* pDC, unsigned int line);
	void UpdateTextLine();
	void SelectionToString(std::string& newstr);
	void StringToVector(std::vector<std::string>& strs, const char* cp);
	void StringToClipboard(const std::string& str);
	void InsertStringsIntoDocument(const std::vector<std::string>& strs, bool bterminate = true);
	void UpdateTitle(CDocument* pDoc);

	void CaretDown();
	void CaretUp();
	void CaretRight();
	void CaretLeft();
	void WordRight();
	void WordLeft();
	void LineEnd();
	void LineHome();

	void DeleteCharacter();
	void DeleteWord();
	void InsertCharacter(char ch, bool bterminate = true);
	void SplitLine();
	void InsertTab();
	void DeleteSelection(bool changecaret = true, bool bterminate = true);

	CPoint ScreenToCharCoordinates(CPoint pt);
	CPoint CharToScreenCoordinates(CPoint pt);

	void FindReplace(bool findonly);
	bool isword(const char* cp, int col, int len);
	void FindTextForward();
	void FindTextBackward();
	void FindNext();

	void DeleteMacros();
	void RecordCommand(MacroCommand* pCmd);

	void SetScrollPosition(int nBar, UINT nPos);
	void VScrollTo(UINT nPos);
	void HScrollTo(UINT nPos);
	void ScrollDown();
	void ScrollUp();
	void ScrollLeft();
	void ScrollRight();
	void PageHome();
	void PageEnd();
	void PageUp();
	void PageDown();
	void PageLeft();
	void PageRight();
	void AdjustScroll();
	void AdjustScrollQuietly();
	void CurrentLineIntoView();
	void TestSelection();
	void TestMarking();

	void SetCaretPosition(bool bmoveright = false, bool frompaging = false);

	int vscrollsize() const			// pixel width for each vertical scroll action
		{ return m_fontheight; }
	int hscrollsize() const			
		{ return m_fontwidth; }	// pixel width for each horizontal scroll action
	int vpagesize() const
		{ return (m_windowrows / m_fontheight) * m_fontheight; }	// pixel width for each vertical page action
	int hpagesize() const
		{ return hscrollsize() * 8; }	// pixel width for each horizontal page action

	void AutoIndent();

	// --- brace matching functions
	bool IsSpecificBrace(char ch, int row, int col);
	bool IsLeftBrace(int row, int col)
		{ return IsSpecificBrace('{', row, col); }
	bool IsRightBrace(int row, int col)
		{ return IsSpecificBrace('}', row, col); }
	bool IsEitherBrace(int row, int col)
		{ return !IsContextElement(row, col) && (IsLeftBrace(row, col) || IsRightBrace(row, col)); }
	bool IsContextElement(int row, int col);
	bool IsCharLiteral(int row, int col);

	void Beautify();

protected: // create from serialization only
	CEditorView();
	DECLARE_DYNCREATE(CEditorView)

	unsigned int fontheight() const { return m_fontheight; }
	unsigned int fontwidth() const  { return m_fontwidth;  }
	void PadMargin();
	void BuildContextHighlightTable();

// Attributes
public:
	CEditorDoc* GetDocument();

// Operations
public:
	// ------ functions to support marked selections
	void InvertDisplayRow(unsigned int row, unsigned int begincol, unsigned int endcol);
	void InvertDisplayItems(ViewIndex begin, ViewIndex end);

	// ------ functions to support view contexts for undo/redo
	void GetCurrentPosition(unsigned int& toprow, unsigned int& leftcolumn, unsigned int& row, unsigned int& column) const;
	void SetCurrentPosition(unsigned int toprow, unsigned int leftcolumn, unsigned int row, unsigned int column);

	void SetLineColumn(int line, int column)	// one-based line and column
	{
		m_currentrow = line > 0 ? line-1 : 0;
		m_currentcolumn = column > 0 ? column-1 : 0;
		CurrentLineIntoView();
	}

	int CurrentLineNumber() const
		{ return m_currentrow + 1; }	// one-based text line number
	int CurrentColumn() const
		{ return m_currentcolumn + 1; }	// one-based text column

	// --- return one-based top text line of display
	unsigned int TopLine() const
		{ return (m_toprow / m_fontheight) + 1; }

	// --- return top pixel line of display
	unsigned int TopRow() const
		{ return m_toprow; }

	bool IsFindText() const
		{ return m_findtext.length() != 0; }

	bool IsSelectionMarked() const;
	std::string SelectedText();

	bool IsRecording() const
		{ return m_recording; }

	void RebuildFont();

	bool ExtendEditor(UINT nChar);	// for adding editor keyboard functions
	void ResetKeywords()
		{ keywords.clear(); m_buildcomments = true; Invalidate(false); }
	bool CanMatchBraces()
		{ return IsEitherBrace(m_currentrow, m_currentcolumn); }

// Overrides
	// ClassWizard generated virtual function overrides
	//{{AFX_VIRTUAL(CEditorView)
	public:
	virtual void OnDraw(CDC* pDC);  // overridden to draw this view
	virtual BOOL PreCreateWindow(CREATESTRUCT& cs);
	virtual BOOL Create(LPCTSTR lpszClassName, LPCTSTR lpszWindowName, DWORD dwStyle, const RECT& rect, CWnd* pParentWnd, UINT nID, CCreateContext* pContext = NULL);
	protected:
	virtual BOOL OnPreparePrinting(CPrintInfo* pInfo);
	virtual void OnBeginPrinting(CDC* pDC, CPrintInfo* pInfo);
	virtual void OnEndPrinting(CDC* pDC, CPrintInfo* pInfo);
	virtual void OnPrint(CDC* pDC, CPrintInfo* pInfo);
	virtual void OnActivateView(BOOL bActivate, CView* pActivateView, CView* pDeactiveView);
	//}}AFX_VIRTUAL
	// void OnUpdate(CView* pSender, LPARAM lHint,CObject* pHint);

// Implementation
public:
	virtual ~CEditorView();
#ifdef _DEBUG
	virtual void AssertValid() const;
	virtual void Dump(CDumpContext& dc) const;
	void DumpContextTable();
#endif

//protected:

// Generated message map functions
//protected:
	//{{AFX_MSG(CEditorView)
	afx_msg void OnVScroll(UINT nSBCode, UINT nPos, CScrollBar* pScrollBar);
	afx_msg void OnHScroll(UINT nSBCode, UINT nPos, CScrollBar* pScrollBar);
	afx_msg void OnKillFocus(CWnd* pNewWnd);
	afx_msg void OnSetFocus(CWnd* pOldWnd);
	afx_msg void OnLButtonDown(UINT nFlags, CPoint point);
	afx_msg void OnLButtonDblClk(UINT nFlags, CPoint point);
	afx_msg void OnKeyDown(UINT nChar, UINT nRepCnt, UINT nFlags);
	afx_msg void OnKeyUp(UINT nChar, UINT nRepCnt, UINT nFlags);
	afx_msg void OnChar(UINT nChar, UINT nRepCnt, UINT nFlags);
	afx_msg void OnLButtonUp(UINT nFlags, CPoint point);
	afx_msg void OnMouseMove(UINT nFlags, CPoint point);
	afx_msg void OnTimer(UINT nIDEvent);
	afx_msg void OnUpdateEditCut(CCmdUI* pCmdUI);
//	afx_msg void OnUpdateEditCopy(CCmdUI* pCmdUI);
	afx_msg void OnUpdateEditPaste(CCmdUI* pCmdUI);
	afx_msg void OnEditCut();
	afx_msg void OnEditCopy();
	afx_msg void OnEditPaste();
	afx_msg void OnUpdateEditClear(CCmdUI* pCmdUI);
	afx_msg void OnEditClear();
	afx_msg void OnEditSelectAll();
	afx_msg void OnEditFind();
	afx_msg void OnEditReplace();
	afx_msg void OnEditUndo();
	afx_msg void OnEditRedo();
	afx_msg void OnRecord();
	afx_msg void OnStopRecord();
	afx_msg BOOL OnSetCursor(CWnd* pWnd, UINT nHitTest, UINT message);
	afx_msg void OnUpdatePlayback(CCmdUI* pCmdUI);
	afx_msg void OnUpdateRecord(CCmdUI* pCmdUI);
	afx_msg void OnUpdateStopRecord(CCmdUI* pCmdUI);
	afx_msg void OnPlayback();
	afx_msg void OnFileOpen();
	afx_msg void OnFilePrint();
	afx_msg void OnUpdateEditFindNext(CCmdUI* pCmdUI);
	afx_msg void OnEditFindNext();
	afx_msg void OnUpdateFileSaveAll(CCmdUI* pCmdUI);
	afx_msg void OnUpdateBracematch(CCmdUI* pCmdUI);
	afx_msg void OnBracematch();
	afx_msg void OnUpdateBeautify(CCmdUI* pCmdUI);
	afx_msg void OnBeautify();
	//}}AFX_MSG

	afx_msg LONG OnFindReplace(WPARAM wParam, LPARAM lParam);

	DECLARE_MESSAGE_MAP()
};


#ifndef _DEBUG  // debug version in EditorView.cpp
inline CEditorDoc* CEditorView::GetDocument()
					{ return (CEditorDoc*)m_pDocument; }
#endif

namespace sel = DDJCProgrammingColumnSelection;

class EditorSelection : public sel::Selection<ViewIndex, CEditorView>
{
public:
	EditorSelection(CEditorView& view);
};

inline bool CEditorView::IsSelectionMarked() const
{
	ASSERT(m_pEditorSelection != 0);
	return m_pEditorSelection->IsSelectionMarked();
}


class MacroCommand {
protected:
	CEditorView* pView;
public:
	MacroCommand(CEditorView* pV = 0) : pView(pV) { }
	virtual ~MacroCommand() { }
	virtual void PlayBack() = 0;
};

class KeyDown : public MacroCommand	{
	UINT key;
public:
	KeyDown(CEditorView* pV, UINT ky) : MacroCommand(pV), key(ky) { }
	void PlayBack()
		{ pView->OnKeyDown(key, 0, 0); }
};

class KeyUp : public MacroCommand	{
	UINT key;
public:
	KeyUp(CEditorView* pV, UINT ky) : MacroCommand(pV), key(ky) { }
	void PlayBack()
		{ pView->OnKeyUp(key, 0, 0); }
};

class Char : public MacroCommand	{
	UINT key;
public:
	Char(CEditorView* pV, UINT ky) : MacroCommand(pV), key(ky) { }
	void PlayBack()
		{ pView->OnChar(key, 0, 0); }
};
/////////////////////////////////////////////////////////////////////////////

//{{AFX_INSERT_LOCATION}}
// Microsoft Developer Studio will insert additional declarations immediately before the previous line.

#endif // !defined(AFX_EDITORVIEW_H__408454CF_548A_11D2_A779_00C0A80034F0__INCLUDED_)
